#!/bin/sh
#
# Script name:   nvtrapcf.sh
#
# Description:   This script adds the CAU specific traps to
#                IBM AIX SystemView NetView/6000 (R) V2R1 by use of the
#		 "addtrap" command.  
#
# Restrictions:  This script is written only for IBM AIX SystemView
#                NetView/6000 (R) V2R1 running on the IBM RISC System/6000.
#

myname=$0

ADDTRAP=/usr/OV/bin/addtrap
TRAPD_CONF_FILE="/usr/OV/conf/trapd.conf"
DFLT_SAVE_EXT="preCau"

ENTERPRISE_ID="1.3.6.1.4.1.2.6.15"
ENTERPRISE_NAME="ibm8230"

MIB_NAME="IBM 8230 MIB"
PROD_NAME="IBM 8230-3"

PROD="IBM_8230_"
counter=0

DFLT_TRAP_SEV="1"
DFLT_SOURCE_ID="A"
DFLT_STATUS_TYPE="0"
DFLT_CATEGORY="Status Events"

#
# get_y_or_n()
#
# Solicit a yes or no answer to a question.  Do not return until either
# "y" or "n" is entered.
#
get_y_or_n()
{
	unset ans

	ans="X"
	until [ $ans = "y" -o $ans = "n" ]
	do
		echo -n "$1 [y/n] "
		read ans
		if [ -z "$ans" ]
		then
			ans="X"
		fi
	done

	if [ $ans = "y" ]
	then
		return 1
	else
		return 0
	fi
}

#
# verify_req_user()
#
# Verify that the user of this script has root permission.  If not, we
# can't go on.
#
verify_req_user()
{
	useris=`whoami`

	if [ "$useris" != "root" ]
	then
		echo
		echo "You must have root permission to run this program"
		echo
		exit 1
	fi
}

#
# verify_command()
#
# Verify that the command necessary to do what we want to do is available.
#
verify_command()
{

	if [ ! -x $ADDTRAP ]
	then
		echo
		echo "The program $ADDTRAP is required to run"
		echo "$myname.  This program could not be found."
		echo "Ensure that that IBM AIX SystemView NetView/6000 is"
		echo "installed correctly."
		echo
		exit 1
	fi
}

#
# verify_intentions()
#
# Make sure that the user knows what we are going to do and that he/she
# really wants to do it.
#
verify_intentions()
{
	echo
	echo "This program will do trap customization of IBM AIX SystemView"
	echo "NetView/6000 for the enterprise specific traps in the"
	echo "$MIB_NAME.  Prior to doing this, the original trap"
	echo "configuration will be saved."
	echo

	if get_y_or_n "Are you sure you want to continue?"
	then
		echo
		echo "Exiting without making any changes"
		exit 1
	fi
}

#
# save_orig_file()
#
# Preserve the original version of the trapd.conf file before we go on.
#
save_orig_file()
{
	save_file=$TRAPD_CONF_FILE.$DFLT_SAVE_EXT
	if [ -f $save_file ]
	then	
		echo
		echo "It appears that the file \"$TRAPD_CONF_FILE\" has already"
		echo "been saved.  It will not be saved again."
	else
		cp -p $TRAPD_CONF_FILE $save_file
		echo
		echo "The original version of the file \"$TRAPD_CONF_FILE\" has"
		echo "been saved as $save_file."
	fi
}

#
# add_cau_trap()
#
# Add a cau specific trap.
#
add_cau_trap()
{
	generic=$1
	specific=$2
	text="$3"


	echo
	echo "Adding trap generic $generic specific $specific"

	$ADDTRAP -n $ENTERPRISE_NAME -i $ENTERPRISE_ID -l $PROD$counter \
	-g $generic -s $specific -S $DFLT_TRAP_SEV -o $DFLT_SOURCE_ID \
	-t $DFLT_STATUS_TYPE -c "$DFLT_CATEGORY" -f - -F "$text"

	counter=`expr $counter + 1`
}

#
# Main part of script.
#

#
# Sanity checks.
#
verify_req_user
verify_command
verify_intentions

#
# Keep original file
#
save_orig_file

#
# Add all of the traps
#
add_cau_trap 0 0 "$PROD_NAME Up with Possible Changes (ColdStart Trap)"

add_cau_trap 6 1 "$PROD_NAME Backup Path Status Change, New Status: \$1"

add_cau_trap 6 2 "$PROD_NAME Wrap Status Change, New Status: \$1"

add_cau_trap 6 3 "$PROD_NAME Internal Error Code is: \$1"

add_cau_trap 6 4 "$PROD_NAME Module Status Change, ModLocation: \$1 ModOperStatus: \$2 ModAdminStatus: \$3 ModFaultStatus: \$4"

add_cau_trap 6 5 "$PROD_NAME Lobe Status Change, Module: \$1 Lobe: \$2 LobeOperStatus: \$3 LobeAdminStatus: \$4 LobeInsertStatus: \$5 LobeFaultStatus: \$6 LobeMaxMACsExceeded: \$7"

add_cau_trap 6 6 "$PROD_NAME Removal Ignored"

add_cau_trap 6 7 "$PROD_NAME Configuration Changed from new IP Address: \$1"

add_cau_trap 6 8 "$PROD_NAME Lobe Configuration Change, Module: \$1 Lobe: \$2 MACAddrs: \$3 MacType: \$4"

add_cau_trap 6 9 "$PROD_NAME Code Update Complete"

add_cau_trap 6 10 "$PROD_NAME Module Removed, Module: \$1"

add_cau_trap 6 11 "$PROD_NAME NAUN Address Changed, New NAUN: \$1"

add_cau_trap 6 12 "$PROD_NAME Remote LAM Control Link Status Change, ModLocation: \$1 Control Link Status: \$2 RLAM Type: \$3 Number of Lobes: \$4"

add_cau_trap 6 13 "$PROD_NAME Remote LAM Data Cable Status Change, ModLocation: \$1 Data Cable Status: \$2"

echo
echo "Trap customization complete"

exit 0
